
/* 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                        AY-3-8912 SYNTHESIZER CHIP OPERATION
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        3 Voice Synthesizer controlled through 16 8-bit registers and 3 Control lines

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~Arduino Pin Assignments:~~~~~~~~~~~~~~~~~~~~~~~~~~~~

        Control lines are BC1 (Arduino pin D4) and BC2 (Arduino Pin  D5), BDIR (Arduino pin D7)
        Data lines are D3/D2/D13/D12/D11/D10/D9/D8 (higher bits D3,D2 + lower 6 bits on PORTB)
        D6 reserved for TONE
        D0 and D1 reserved for serial communication with host computer (i.e. program loading)
        AY Reset tied to Arduino Reset
        
        A0, A1, A2  Digital Inputs for 3 Switches (turn on pullup resistor)
        A3, A4, A5  Analog Inputs for 3 Slide Pots (0 to 5 volts)

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~Operation:~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Normal Inactive Rest State -- All 3 control lines (BC1, BC2, BDIR) are LOW, (010 is also Inactive)
    
    Load Address 0-15 - D2->LOW, D3->LOW, Write Address onto PORTB (D8-D13), 
                        with BC1=0 and BC2=0, pulse BDIR high 
                        
    Load 8-bit Data - Put Data on D2/D3/D13/D12/D11/D10/D9/D8 (higher 2 bits D2,D3 -- lower 6 bits PORTB)
                        with BC1=0 and BC2=1, pulse BDIR high 
                        
    Read Data at Address - Only used for test purposes.
                       Load Address you want to read.  Make data lines D2/D3/D13/D12/D11/D10/D9/D8 Inputs
                       From Normal Inactive State, BC2 HIGH, pulse BC1 high, Read Data lines

      See Spec Sheets for the functions of the 16 registers.  Three voices A, B, and C plus pitched noise.
      Amplitude control direct from Computer (4-bits, Mode=0) or from Synth's Envelope Generator (Mode=1).

~~~~~~~~~~~~~~~~~~~~~~~~~Basic Control Functions:~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            loadAddress(int address)  Load control register address 0 through 13
            loadData(int data)   Load 8-bit data value into the register set by loadAddress()
            loadDataShort(int data) Load 4 to 6-bit data value (slightly faster than loadData)
            
 ~~~~~~~~~~~~~~~~~~~~~~~~~~~~Synthesizer Functions: ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            
            ldSynth(int address, int data) General Load 4-bit Control Register Address then 8-bit data
            
            ldFineTuneA(int data)    8-bit fine tune Oscillator A
            ldFineTuneB(int data)    8-bit fine tune Oscillator B
            ldFineTuneC(int data)    8-bit fine tune Oscillator C
            
            ldCourseTuneA(int data)  4-bit course tune Oscillator A
            ldCourseTuneB(int data)  4-bit course tune Oscillator B
            ldCourseTuneC(int data)  4-bit course tune Oscillator C
            
            ldNoisePeriod(int data)  5-bit Noise period tune
            
            ldEnable(int data)   Low 1-bit enable for each sound source, Noise C/B/A Oscillator C/B/A 
            
            ldAmpA(int amp)   4-bit amplitude of A if 0-15, else use synth envelope if = 16
            ldAmpB(int amp)   4-bit amplitude of B if 0-15, else use synth envelope if = 16
            ldAmpC(int amp)   4-bit amplitude of C if 0-15, else use synth envelope if = 16
            
            ldEnvFineTune(int data)   8-bit Envelope Period fine tune
            ldEnvCourseTune(int data  8-bit Envelope Period course tune
            ldEnvShape(int data)      4-bit Envelope Shape. Continue/Attack/Alternate/Hold
                         
                            00xx, 1001 Ramp Off, Staying Off
                            1011       Ramp Off then Immediate On, Staying On
                            
                            1101       Ramp On, Staying On
                            01xx, 1111 Ramp On, then Immediate Off, Staying Off
                                                                                  
                            1000 Sawtooth, Ramp Off then Immediate On Repeated
                            1100 Sawtooth, Ramp On then Immediate Off Repeated
                            
                            1010 Triangle, Ramp Off and Ramp On Repeated
                            1110 Triangle, Ramp On and Ramp Off Repeated                     
*/

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      CONSTANTS DEFINED
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

const int BDIR = 7;  // set up names for some Arduino pins
const int BC1 = 4;
const int BC2 = 5;
const int AYD7 = 2;
const int AYD6 = 3;

const int TONE_D = 6;

const int FineTuneA = 0;  // AY Synth Control Registers
const int FineTuneB = 2;
const int FineTuneC = 4;
const int CourseTuneA = 1;
const int CourseTuneB = 3;
const int CourseTuneC = 5;
const int NoisePeriod = 6;
const int Enable = 7;
const int AmpA = 8;
const int AmpB = 9;
const int AmpC = 10;
const int EnvFineTune = 11;
const int EnvCourseTune = 12;
const int EnvShape = 13;

int Slider0 = 0;  //Use Slider0 = analogRead(A3);
int Slider1 = 0;  //Use Slider1 = analogRead(A4);
int Slider2 = 0;  //Use Slider2 = analogRead(A5);
int Button0 = 0;  //Use Button0 = digitalRead(A0);
int Button1 = 0;  //Use Button1 = digitalRead(A1);
int Toggle = 0;  //Use   Toggle = digitalRead(A2);

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                            SETUP
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void setup() {

    Serial.begin(9600);

pinMode(BDIR, OUTPUT);
digitalWrite(BDIR, LOW);
pinMode(BC1, OUTPUT);
digitalWrite(BC1, LOW);
pinMode(BC2, OUTPUT);
digitalWrite(BC2, LOW);
pinMode(AYD7, OUTPUT);
pinMode(AYD6, OUTPUT);

DDRB = B00111111;  // Lower 6-bits are outputs, higher 2 bits left alone as inputs

pinMode(A0, INPUT);  // Setup A0/A1/A2 as switch inputs with pullup resistors
digitalWrite(A0, HIGH);
pinMode(A1, INPUT);
digitalWrite(A1, HIGH);
pinMode(A2, INPUT);
digitalWrite(A2, HIGH);

 // A3/A4/A5 connected to Analog volage pots
                    
pinMode(TONE_D, OUTPUT); // Arduino output to use with TONE
  
} //End of Setup

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      MAIN LOOP
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void loop() {

}  // End of Main Loop


//  ______________________________________________________________________________________
//                Address and Data Load Functions
//  ______________________________________________________________________________________

void loadAddress(int address){
  digitalWrite(AYD7, LOW);
  digitalWrite(AYD6, LOW);
  PORTB = (address & B1111);
  delayMicroseconds(50);
  digitalWrite(BDIR, HIGH);
  delayMicroseconds(50);
  digitalWrite(BDIR, LOW);
}

void loadData(int data){
  if((data & 128) == 0)
  { digitalWrite(AYD7, LOW); } else
  { digitalWrite(AYD7, HIGH); }
  
  if((data & 64) == 0)
  { digitalWrite(AYD6, LOW); } else
  { digitalWrite(AYD6, HIGH); }
    
  PORTB = (data & B111111);

  delayMicroseconds(50);
  digitalWrite(BC2, HIGH);
  digitalWrite(BDIR, HIGH);
  delayMicroseconds(50);
  digitalWrite(BDIR, LOW);
  digitalWrite(BC2, LOW);
}

void loadDataShort(int data){
  PORTB = (data & B111111);
  delayMicroseconds(50);
  digitalWrite(BC2, HIGH);
  digitalWrite(BDIR, HIGH);
  delayMicroseconds(50);
  digitalWrite(BDIR, LOW);
  digitalWrite(BC2, LOW);
}


//  ______________________________________________________________________________________
//                General Control Register Load
//  ______________________________________________________________________________________

void ldSynth(int address, int data){  // Load 4-bit Control Register Address then 8-bit data
  loadAddress(address);
  loadData(data);
}

//  ______________________________________________________________________________________
//                Individual Control Register Load 
//  ______________________________________________________________________________________


void ldFineTuneA(int data){  //8-bit fine tune A
  loadAddress(FineTuneA);
  loadData(data);
}

void ldFineTuneB(int data){  //8-bit fine tune B
  loadAddress(FineTuneB);
  loadData(data);
}

void ldFineTuneC(int data){  //8-bit fine tune C
  loadAddress(FineTuneC);
  loadData(data);
}

void ldCourseTuneA(int data){  //4-bit course tune A
  loadAddress(CourseTuneA);
  loadDataShort(data);
}

void ldCourseTuneB(int data){  //4-bit course tune B
  loadAddress(CourseTuneB);
  loadDataShort(data);
}

void ldCourseTuneC(int data){  //4-bit course tune C
  loadAddress(CourseTuneC);
  loadDataShort(data);
}

void ldNoisePeriod(int data){  //5-bit noise period tune
  loadAddress(NoisePeriod);
  loadDataShort(data);
}

void ldEnable(int data){  //Low 1-bit enable, Noise C/B/A Tone C/B/A
  loadAddress(Enable);
  loadDataShort(data);
}

void ldAmpA(int amp){  //4-bit amplitude of A if 0-15, else use Env if 16 (mode bit high)
  loadAddress(AmpA);
  loadDataShort(amp);
}

void ldAmpB(int amp){  //4-bit amplitude of B if 0-15, else use Env if 16 (mode bit high)
  loadAddress(AmpB);
  loadDataShort(amp);
}

void ldAmpC(int amp){  //4-bit amplitude of C if 0-15, else use Env if 16 (mode bit high)
  loadAddress(AmpC);
  loadDataShort(amp);
}

void ldEnvFineTune(int data){  //8-bit Envelope Period fine tune
  loadAddress(EnvFineTune);
  loadData(data);
}

void ldEnvCourseTune(int data){  //8-bit Envelope Period course tune
  loadAddress(EnvCourseTune);
  loadData(data);
}

void ldEnvShape(int data){  //4-bit Envelope Shape. Continue/Attack/Alternate/Hold
  loadAddress(EnvShape);
  loadDataShort(data);
}

int readAY() {    //Just used to test the AY chip to confirm writing to the 16 control registers
    pinMode(AYD7, INPUT);  // Make Arduino data pins inputs
    pinMode(AYD6, INPUT);
    DDRB = B00000000;

    digitalWrite(BC2, HIGH);   // AY put into state to output data
    digitalWrite(BC1, HIGH);
    
    int result = PINB & B111111;  // Read AY data from addressed register
    if (digitalRead(AYD7) == HIGH)  { result += 128; }
    if (digitalRead(AYD6) == HIGH)  { result += 64; }
    
    digitalWrite(BC2, LOW);  // Return AY back to inactive state
    digitalWrite(BC1, LOW);
    
    pinMode(AYD7, OUTPUT);   // Return Arduino data lines back to outputs
    pinMode(AYD6, OUTPUT);
    DDRB = B00111111;  // Lower 6-bits are outputs, higher 2 bits left alone as inputs

  return result;
}









